/**
 *  @file obex.c
 *
 *  btfs - Bluetooth FileSystemMapping
 *
 *  @author Collin R. Mulliner <collin@betaversion.net>
 *
 *  (c) Collin R. Mulliner
 *
 *  web: www.mulliner.org/bluetooth/btfs.php
 */

/*
 * This file is part of btfs
 *
 * btfs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * btfs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with btfs; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <bluetooth/bluetooth.h>
#include <openobex/obex.h>
#include "btfs.h"

/**
 *  @brief OBEX event handler (callback)
 */
static void eventHandler(obex_t *handle, obex_object_t *obj, int mode, int event, int obex_cmd, int obex_rsp)
{
	int *finished;
	

	switch (event) {
	case OBEX_EV_REQDONE:
		if (obex_cmd == OBEX_CMD_PUT) {
			finished = OBEX_GetUserData(handle);
			*finished = 1;
		}
		break;
	}
}

/**
 *  @brief send file to device
 *
 *  @param btfs pointer to btfs globals
 *  @param dev  pointer to target device
 *
 *  @returns 1 on success and <= 0 on failure
 */
int obexSendFile(btfs *self, bt_device *dev)
{
	obex_t *libObex;
	obex_object_t *object;
	obex_headerdata_t hd;
	char *name = NULL;
	int finished;
	int res = 1;
	
	
	if ((libObex = OBEX_Init(OBEX_TRANS_BLUETOOTH, eventHandler, 0)) == NULL) {
		#ifdef DEBUG
		printf("obex: init failed\n");
		#endif
		res = 0;
		goto free;
	}
		
	if (BtOBEX_TransportConnect(libObex, &self->local, &dev->addr, dev->obex_channel) < 0) {
		#ifdef DEBUG
		printf("obex: TransportConnect() error\n");
		#endif
		res = 0;
		goto free;
	}
	
	if ((object = OBEX_ObjectNew(libObex, OBEX_CMD_PUT)) == NULL) {
		#ifdef DEBUG
		printf("obex: PUT error\n");
		#endif
		res = 0;
		goto free;
	}

	/* Add length header */
	memset(&hd, 0, sizeof(hd));
	hd.bq4 = dev->ops->op.obexPUT.size;
	if (OBEX_ObjectAddHeader(libObex, object, OBEX_HDR_LENGTH, hd, 4, 0) < 0) {
		#ifdef DEBUG
		printf("obex: length header error\n");
		#endif
		res = 0;
		goto free;
	}

	memset(&hd, 0, sizeof(hd));	
	/* Add unicode name header*/
	name = malloc((strlen(dev->ops->op.obexPUT.name)+1)*2);
	if (OBEX_CharToUnicode(name, dev->ops->op.obexPUT.name, 
			(strlen(dev->ops->op.obexPUT.name)+1)*2) < 0) {
		#ifdef DEBUG
		printf("obex: char to unicode error\n");
		#endif
		res = 0;
		goto free;
	}
	
	hd.bs = name;
	if (OBEX_ObjectAddHeader(libObex, object, OBEX_HDR_NAME, hd, 
			strlen(dev->ops->op.obexPUT.name)*2, 0) < 0) {
		#ifdef  DEBUG
		printf("obex: name header error\n");
		#endif
		res = 0;
		goto free;
	}

	/* Add body header*/
	memset(&hd, 0, sizeof(hd));
	hd.bs = dev->ops->op.obexPUT.data;

	if (OBEX_ObjectAddHeader(libObex, object, OBEX_HDR_BODY, hd, dev->ops->op.obexPUT.size, 0) < 0) {
		#ifdef DEBUG
		printf("obex: body header error\n");
		#endif
		res = 0;
		goto free;
	}

	finished = 0;
	OBEX_SetUserData(libObex, &finished);
		
	if (OBEX_Request(libObex, object) >= 0) {
		while (finished == 0) {
			if (OBEX_HandleInput(libObex, 1) == -1) {
				#ifdef DEBUG
				printf("obex: handle input error\n");
				#endif
				res = 0;
				break;
			}
		}
	}
	else {
		#ifdef DEBUG
		printf("obex: request cmd error\n");
		#endif
		res = 0;
	}

free:
	if (libObex != NULL) OBEX_TransportDisconnect(libObex);
	
	if (name != NULL) free(name);
	free(dev->ops->op.obexPUT.data);
	free(dev->ops);
	dev->ops = NULL;
	
	if (libObex != NULL) OBEX_Cleanup(libObex);
	
	return(res);
}
