/**
 *  @file btfs.c
 *
 *  btfs - Bluetooth FileSystemMapping
 *
 *  @author Collin R. Mulliner <collin@betaversion.net>
 *
 *  (c) Collin R. Mulliner
 *
 *  web: www.mulliner.org/bluetooth/btfs.php
 */

/*
 * This file is part of btfs
 *
 * btfs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * btfs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with btfs; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef linux
#define _XOPEN_SOURCE 500
#endif

#define _GNU_SOURCE
#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <sys/statfs.h>
#include <pthread.h>

#include "btfs.h"
#include "bt_util.h"
#include "util.h"
#include "obex.h"

/* btfs globals (init in main) */
static btfs *globals;

/* virtual tree */
static const char *PATH_DEVICE = "/DEVICES";
static const char *PATH_CFG = "/CFG";
static const char *PATH_OPUSH = "/OPUSH";
static const char *PATH_CFG_INQUIRY_DURATION = "inquiry_duration";
static const char *PATH_CFG_CACHE_TIME = "cache_time";
static const char *PATH_CFG_SCAN_NOW = "SCAN_NOW";

static int xmp_getattr(const char *path, struct stat *stbuf)
{
	int res = 0;
	char *fn;
	char dir[256];
	bt_device *dev;
	

	#ifdef DEBUG
	printf("getattr: path=%s\n", path);
	#endif

	memset(stbuf, 0, sizeof(struct stat));
	
	if(strcmp(path, "/") == 0) {
		stbuf->st_mode = S_IFDIR | 0755;
		stbuf->st_nlink = 2;
	}
	else if (strcmp(path, PATH_DEVICE) == 0) {
		stbuf->st_mode = S_IFDIR | 0777;
		stbuf->st_nlink = 1;
		stbuf->st_size = strlen(path);
	}
	else if (strcmp(path, PATH_CFG) == 0) {
		stbuf->st_mode = S_IFDIR | 0777;
		stbuf->st_nlink = 1;
		stbuf->st_size = strlen(PATH_CFG);
	}
	else if (strcmp(path, PATH_OPUSH) == 0) {
		stbuf->st_mode = S_IFDIR | 0777;
		stbuf->st_nlink = 1;
		stbuf->st_size = strlen(PATH_OPUSH);
	}
	else if (strstr(path, PATH_CFG) > 0) {
		fn = getFileName(path);
		if (strcmp(fn, PATH_CFG_INQUIRY_DURATION) == 0 || 
				strcmp(fn, PATH_CFG_CACHE_TIME) == 0 || 
				strcmp(fn, PATH_CFG_SCAN_NOW) == 0) {
				
		stbuf->st_mode = S_IFREG | 0666;
		stbuf->st_nlink = 1;
		stbuf->st_size = strlen(path);
		}
		else return(-ENOENT);
	}
	else if (strstr(path, PATH_DEVICE) > 0) {
		pthread_rwlock_rdlock(&globals->device_list.lock);
	
		fn = getFileName(path);
		if (isValidAddr(globals, fn) == 1) {
			stbuf->st_mode = S_IFDIR | 0777;
			stbuf->st_nlink = 1;
			stbuf->st_size = strlen(path);
		}
		else {
			res = -ENOENT;
		}
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else if (strstr(path, PATH_OPUSH) > 0) {
		fn = getFileName(path);
		
		pthread_rwlock_rdlock(&globals->device_list.lock);
		
		if (isValidName(globals, fn) == 1) {
			stbuf->st_mode = S_IFDIR | 0777;
			stbuf->st_nlink = 1;
			stbuf->st_size = strlen(path);
		}
		else {
			memset(dir, 0, 256);
			if (getDirName(path, dir) == 1) {
				#ifdef DEBUG
				printf("dir=%s\n", dir);
				#endif
				fn = getFileName(dir);
				#ifdef DEBUG
				printf("fn=%s\n", fn);
				#endif
				if (getDeviceByPath(globals, fn, &dev) == 1) {
					fn = getFileName(path);
					#ifdef DEBUG
					printf("fn=%s\n", fn);
					#endif
					if (dev->ops != NULL) {
						#ifdef DEBUG
						printf("ops != NULL\n");
						#endif
						if (dev->ops->otype == obexPUT) {
							#ifdef DEBUG
							printf("name=%s\n", dev->ops->op.obexPUT.name);
							#endif
							if (strcmp(dev->ops->op.obexPUT.name, fn) == 0) {
								stbuf->st_mode = S_IFREG | 0666;
								stbuf->st_nlink = 1;
								stbuf->st_size = strlen(path);
								res = 0;
								goto out;
							}
						}
					}
				}
			}
			
			res = -ENOENT;
		}
out:
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else {
		res = -ENOENT;
	}
	
	if (res != -ENOENT) {
		stbuf->st_uid = getuid();
		stbuf->st_gid = getgid();
	}

	return(res);
}

static int xmp_getdir(const char *path, fuse_dirh_t h, fuse_dirfil_t filler)
{
	char addr[18];
	int i;
	char *fn;
	time_t tmp;
	int res = 0;


	tmp = time(NULL);

	if (strcmp(path, "/") == 0) {
		filler(h, ".", 4);
		filler(h, "..", 4);
		filler(h, PATH_CFG + 1, 4);
		filler(h, PATH_OPUSH + 1, 4);
		filler(h, PATH_DEVICE + 1, 4);
	}
	else if (strcmp(path, PATH_CFG) == 0) {
		filler(h, ".", 4);
		filler(h, "..", 4);
		filler(h, PATH_CFG_SCAN_NOW, 0);
		filler(h, PATH_CFG_INQUIRY_DURATION, 0);
		filler(h, PATH_CFG_CACHE_TIME, 0);
	}
	else if (strcmp(path, PATH_OPUSH) == 0) {
	
		pthread_rwlock_rdlock(&globals->device_list.lock);
	
		if (tmp >= (globals->device_list.when + globals->cache_time)) {
			pthread_rwlock_unlock(&globals->device_list.lock);
			
			bt_scan(globals);
			globals->device_list.when = time(NULL);
			pthread_rwlock_rdlock(&globals->device_list.lock);
		}
	
		filler(h, ".", 4);
		filler(h, "..", 4);
		for (i = 0; i < globals->device_list.num; i++) {
			if (globals->device_list.list[i].obex_channel != -1) {
				filler(h, globals->device_list.list[i].name, 4);
			}
		}
		
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else if (strcmp(path, PATH_DEVICE) == 0) {
	
		pthread_rwlock_rdlock(&globals->device_list.lock);
	
		if (tmp >= (globals->device_list.when + globals->cache_time)) {
			pthread_rwlock_unlock(&globals->device_list.lock);
			
			bt_scan(globals);
			globals->device_list.when = time(NULL);
			pthread_rwlock_rdlock(&globals->device_list.lock);
		}
	
		filler(h, ".", 4); 
		filler(h, "..", 4);
		for (i = 0; i < globals->device_list.num; i++) {
 			ba2str(&globals->device_list.list[i].addr, addr);
			filler(h, addr, 4);
		}
		
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else if (strstr(path, PATH_DEVICE) > 0) {
		pthread_rwlock_rdlock(&globals->device_list.lock);
		
		fn = getFileName(path);
		if (isValidAddr(globals, fn) == 1) {
			filler(h, ".", 4);
			filler(h, "..", 4);
		}
		else res = -ENOENT;
		
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else if (strstr(path, PATH_OPUSH) > 0) {
		pthread_rwlock_rdlock(&globals->device_list.lock);
	
		fn = getFileName(path);
		if (isValidName(globals, fn) == 1) {
			filler(h, ".", 4);
			filler(h, "..", 4);
		}
		else res = -ENOENT;
		
		pthread_rwlock_unlock(&globals->device_list.lock);
	}
	else {
		return(-ENOENT);
	}

	return(res);
}

static int xmp_mknod(const char *path, mode_t mode, dev_t rdev)
{
	char dir[256];
	char *fn;
	bt_file_operation *fo;
	bt_device *dev;
	int res = 0;
	

	#ifdef DEBUG
	printf("mknod: path=%s\n", path);
	#endif
	
	memset(dir, 0, 256);
	getDirName(path, dir);
	
	#ifdef DEBUG
	printf("mknod: path=%s\n", dir);
	#endif

	pthread_rwlock_wrlock(&globals->device_list.lock);
	
	fn = getFileName(dir);
	if (isValidName(globals, fn) == 1) {
		fo = malloc(sizeof(bt_file_operation));
		memset(fo, 0, sizeof(bt_file_operation));
		
		fn = getFileName(path);
		#ifdef DEBUG
		printf("mknod: name=%s\n", fn);
		#endif
		
		fo->otype = obexPUT;
		strncpy(fo->op.obexPUT.name, fn, strlen(fn));
		fo->op.obexPUT.size = -1;
		fo->op.obexPUT.data = NULL;
		
		fn = getFileName(dir);
		if (getDeviceByPath(globals, fn, &dev) == 1) {
			addFileOperation(dev, fo);
			#ifdef DEBUG
			printf("mknod: addFile done\n");
			#endif
		}
	}
	else res = -errno;
	
	pthread_rwlock_unlock(&globals->device_list.lock);
		
	return(res);
}

static int xmp_open(const char *path, int flags)
{
	char *fn;
	char dir[256];
	bt_device *dev;	
	int res = -errno;
	

	#ifdef DEBUG
	printf("open: path=%s\n", path);
	#endif
	
	memset(dir, 0, 256);
	getDirName(path, dir);

	pthread_rwlock_wrlock(&globals->device_list.lock);
		
	if (strstr(dir, PATH_CFG) != NULL) {
		fn = getFileName(path);
		if (strcmp(fn, PATH_CFG_SCAN_NOW) == 0) {
			pthread_rwlock_unlock(&globals->device_list.lock);
			bt_scan(globals);
			pthread_rwlock_wrlock(&globals->device_list.lock);
			globals->device_list.when = time(NULL);
			res = 0;
		}
		res = 0;	
	}
	else {
		fn = getFileName(dir);
	
		if (getDeviceByPath(globals, fn, &dev) == 1) {
			if (dev->ops != NULL) {
				if (dev->ops->otype == obexPUT) {
					if (dev->ops->op.obexPUT.state == 0) {
						dev->ops->op.obexPUT.state = 1;
						res = 0;
					}
				}
			}
		}
	}
	
	pthread_rwlock_unlock(&globals->device_list.lock);
	
	return(res);
}

static int xmp_release(const char *path, int flags)
{
	char *fn;
	char dir[256];
	bt_device *dev;	
	int res = -errno;
	

	#ifdef DEBUG
	printf("close: path=%s\n", path);
	#endif
	
	pthread_rwlock_wrlock(&globals->device_list.lock);
	
	memset(dir, 0, 256);
	getDirName(path, dir);
	fn = getFileName(dir);
	if (getDeviceByPath(globals, fn, &dev) == 1) {
		if (dev->ops != NULL) {
			if (dev->ops->otype == obexPUT) {
				if (dev->ops->op.obexPUT.state == 1) {
					dev->ops->op.obexPUT.state = 0;

					// kickoff obex PUT here
					#ifdef DEBUG
					printf("sending...");
					#endif
					obexSendFile(globals, dev);
					#ifdef DEBUG
					printf("done\n");
					#endif
					
					res = 0;
				}
			}
		}
	}

	pthread_rwlock_unlock(&globals->device_list.lock);
	
	return(res);
}

static int xmp_read(const char *path, char *buf, size_t size, off_t offset)
{
	char dir[256];
	char *fn;
	int res = 0;
	

	#ifdef DEBUG
	printf("read: path=%s\n", path);
	printf("size=%d offset=%d\n", size, offset);
	#endif
	
	memset(dir, 0, 256);
	getDirName(path, dir);
	fn = getFileName(path);
	#ifdef DEBUG
	printf("dir=%s fn=%s\n", dir, fn);
	#endif
	
	if (strcmp(dir, PATH_CFG) == 0) {
		pthread_mutex_lock(&globals->cfg_lock);
	
		if (strcmp(fn, PATH_CFG_INQUIRY_DURATION) == 0) {
			memset(dir, 0, 256);
			sprintf(dir, "%d", globals->inquiry_duration);
			memcpy(buf, dir, strlen(dir));
			res = strlen(dir);
		}
		else if (strcmp(fn, PATH_CFG_CACHE_TIME) == 0) {
			memset(dir, 0, 256);
			sprintf(dir, "%d", globals->cache_time);
			memcpy(buf, dir, strlen(dir));
			res = strlen(dir);
		}
		pthread_mutex_unlock(&globals->cfg_lock);
	}
	
	return(res);
}

static int xmp_write(const char *path, const char *buf, size_t size, off_t offset)
{
	char *fn;
	char dir[256];
	bt_device *dev;
	char *tmp;
	int res = -errno;
	

	#ifdef DEBUG
	printf("write: path=%s\n", path);
	#endif
	
	memset(dir, 0, 256);
	getDirName(path, dir);
	fn = getFileName(path);

	#ifdef DEBUG
	printf("dir=%s fn=%s\n", dir, fn);
	#endif
	
	if (strcmp(dir, PATH_CFG) == 0) {
		int tmp;
		
		pthread_mutex_lock(&globals->cfg_lock);
		
		if (strcmp(fn, PATH_CFG_INQUIRY_DURATION) == 0) {
			memset(dir, 0, 256);
			sscanf(buf, "%d", &tmp);
			if (tmp != 0) globals->inquiry_duration = tmp;
			res = 0;
		}
		else if (strcmp(fn, PATH_CFG_CACHE_TIME) == 0) {
			memset(dir, 0, 256);
			sscanf(buf, "%d", &tmp);
			if (tmp != 0) globals->cache_time = tmp;
			res = 0;
		}
		
		pthread_mutex_unlock(&globals->cfg_lock);
	}
	else {
	fn = getFileName(dir);
	pthread_rwlock_wrlock(&globals->device_list.lock);
	
	if (getDeviceByPath(globals, fn, &dev) == 1) {
		if (dev->ops != NULL) {
			if (dev->ops->otype == obexPUT) {
				if (dev->ops->op.obexPUT.state == 1) {
					if (dev->ops->op.obexPUT.data == NULL) {
						dev->ops->op.obexPUT.data = malloc(size + offset);
						if (dev->ops->op.obexPUT.data == NULL) return(-errno);
						else {
							memcpy(dev->ops->op.obexPUT.data+offset, buf, size);
							dev->ops->op.obexPUT.size = offset + size;
							res = 0;
						}
					}
					else {
						if (dev->ops->op.obexPUT.size >= (offset + size)) {
							memcpy(dev->ops->op.obexPUT.data + offset, buf, size);
							res = 0;
						}
						else {
							tmp = malloc(offset + size);
							if (tmp == NULL) return(-errno);
							memcpy(tmp, dev->ops->op.obexPUT.data, dev->ops->op.obexPUT.size);
							memcpy(tmp + offset, buf, size);
							dev->ops->op.obexPUT.size = offset + size;
							free(dev->ops->op.obexPUT.data);
							dev->ops->op.obexPUT.data = tmp;
							res = 0;
						}
					}
				}
			}
		}
	}
		pthread_rwlock_unlock(&globals->device_list.lock);
	}

	return(size);
}

static int xmp_chmod(const char *path, mode_t mode)
{
	#ifdef DEBUG
	printf("chmod: path=%s\n", path);
	#endif
	return(0);
}

static int xmp_chown(const char *path, uid_t uid, gid_t gid)
{
	#ifdef DEBUG
	printf("chown: path=%s\n", path);
	#endif
	return(0);
}

static int xmp_truncate(const char *path, off_t size)
{
	#ifdef DEBUG
	printf("truncate: path=%s\n", path);
	#endif
	return(0);
}

static int xmp_utime(const char *path, struct utimbuf *buf)
{
	#ifdef DEBUG
	printf("utime: path=%s\n", path);
	#endif
	return(0);
}

static struct fuse_operations xmp_oper = {
    .getattr	= xmp_getattr,
    .getdir	= xmp_getdir,
    .mknod	= xmp_mknod,
    .open	= xmp_open,
    .read	= xmp_read,
    .write	= xmp_write,
	 .release = xmp_release,
	 .chmod = xmp_chmod,
	 .chown = xmp_chown,
	 .truncate = xmp_truncate,
	 .utime = xmp_utime
};

int main(int argc, char *argv[])
{
	globals = malloc(sizeof(btfs));
	memset(globals, 0, sizeof(btfs));
	
	bacpy(&globals->local, BDADDR_ANY);
	
	pthread_mutex_init(&globals->cfg_lock, NULL);
	globals->max_devices = 250;
	globals->inquiry_duration = 7;
	globals->cache_time = 30;
	
	pthread_rwlock_init(&globals->device_list.lock, NULL);
	globals->device_list.list = NULL;
	globals->device_list.num = 0;
	
	#ifdef DEBUG
	globals->device_list.when = time(NULL);
	printf("[btfs] DEBUG scanning for devices ... please wait\n");
	bt_scan(globals);
	printf("num devices: %d\n",globals->device_list.num);
	if (globals->device_list.num >= 1) printf("first device name: %s\n", globals->device_list.list[0].name);
	printf("[btfs] ready\n");
	#else
	globals->device_list.when = 0;
	#endif

	fuse_main(argc, argv, &xmp_oper);

	pthread_rwlock_destroy(&globals->device_list.lock);
	pthread_mutex_destroy(&globals->cfg_lock);

	#ifdef DEBUG
	printf("[btfs] exiting\n");
	#endif
	
	return(0);
}
