/**
 *  @file bt_util.c
 *
 *  btfs - Bluetooth FileSystemMapping
 *
 *  @author Collin R. Mulliner <collin@betaversion.net>
 *
 *  (c) Collin R. Mulliner
 *
 *  web: www.mulliner.org/bluetooth/btfs.php
 */

/*
 * This file is part of btfs
 *
 * btfs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * btfs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with btfs; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define _GNU_SOURCE
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>
#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>
#include <bluetooth/rfcomm.h>

#include "btfs.h"

/**
 *  @brief query a device for the RFCOMM OPUSH channel
 *
 *  fire a SDP query for the RFCOMM channel of the OBEX server
 *
 *  @param src     pointer to source bd_addr
 *  @param dst     pointer to destination bd_addr
 *  @param channel pointer where to write the result channel
 *
 *  @returns 1 on success and <= 0 on failure
 */
int sdp_get_channel_opush(bdaddr_t *src, bdaddr_t *dst, int *channel)
{
	uuid_t service;
	sdp_session_t *session;
	sdp_list_t *search, *attrs, *rsp;
	uint16_t attr;
	int err;

	/* build search request */
	sdp_uuid16_create(&service, OBEX_OBJPUSH_SVCLASS_ID);
	search = sdp_list_append(0, &service);
	attr = SDP_ATTR_PROTO_DESC_LIST;
	attrs = sdp_list_append(NULL, &attr);

	/* connect */
	session = sdp_connect(src, dst, SDP_RETRY_IF_BUSY);
	if (!session) return(-1);
	
	/* send request */
	err = sdp_service_search_attr_req(session, search, SDP_ATTR_REQ_INDIVIDUAL, attrs, &rsp);
	/* close connection */
	sdp_close(session);
        
	if (err) return(0);

	/* get rfcomm channel */
	for(; rsp; rsp = rsp->next) {
		sdp_record_t *rec = (sdp_record_t *) rsp->data;
		sdp_list_t *protos;

		if (!sdp_get_access_protos(rec, &protos)) {
			int ch = sdp_get_proto_port(protos, RFCOMM_UUID);
			if (ch > 0) {
				*channel = ch;
				return(1);
			}
		}
	}
	
	return(0);	
}

/**
 *  @brief build the local bt-device database
 *
 *  @param self pointer to btfs globals
 *
 *  @returns 1 on success and <= 0 on failure
 */
int bt_scan(btfs *self)
{
	int dev_id;
	inquiry_info *info = NULL;
	int num_rsp, flags;
	int i, dd, c;
	bt_dev_list tmp_lst;
	
	
	flags = 0;
	
	dev_id = hci_get_route(NULL);
	if (dev_id < 0) return(-1);
	
	num_rsp = hci_inquiry(dev_id, self->inquiry_duration, self->max_devices, NULL, &info, flags);
	
	dd = hci_open_dev(dev_id);
	if (dd < 0) return(-1);
	
	tmp_lst.list = malloc(num_rsp * sizeof(bt_device));
	memset(tmp_lst.list, 0, num_rsp * sizeof(bt_device));
	tmp_lst.num = num_rsp;
	
	for (i = 0; i < num_rsp; i++) {
		memcpy(&tmp_lst.list[i].addr, &(info+i)->bdaddr, sizeof(bdaddr_t));
		tmp_lst.list[i].obex_channel = -1;
		ba2str(&(info+i)->bdaddr, tmp_lst.list[i].bdaddr_str);
		if (hci_read_remote_name(dd, &(info+i)->bdaddr, sizeof(tmp_lst.list[i].name), tmp_lst.list[i].name, 100000) < 0) {
			strcpy(tmp_lst.list[i].name, tmp_lst.list[i].bdaddr_str);
			strcat(tmp_lst.list[i].name, "_noName");
		}
		if (sdp_get_channel_opush(&self->local, &(info+i)->bdaddr, &c)) {
			tmp_lst.list[i].obex_channel = c;
		}
	}
	
	close(dd);
	free(info);

	pthread_rwlock_wrlock(&self->device_list.lock);
	
	/* free old list */
	if (self->device_list.list != NULL) {
		free(self->device_list.list);
		self->device_list.list = NULL;
	}
	
	/* update global list */
	self->device_list.when = time(NULL);
	self->device_list.list = tmp_lst.list;
	self->device_list.num = tmp_lst.num;
	
	pthread_rwlock_unlock(&self->device_list.lock);
	
	return(1);
}
